'use strict';

jest.mock('request-promise-native');
const alpha = require('../')();

test(`the url builder properly builds urls`, () => {
  expect.assertions(7);
  const frags = alpha.util
    .url({
      function: 'a',
      symbol: 'b',
      outputsize: 'c',
      datatype: 'd',
      interval: 'e'
    })
    .toString()
    .split('?')[1]
    .toString()
    .split('&');

  expect(frags.length).toBe(6);
  expect(/^apikey=.*$/.test(frags[0])).toBe(true);
  expect(frags[1]).toBe('function=a');
  expect(frags[2]).toBe('symbol=b');
  expect(frags[3]).toBe('outputsize=c');
  expect(frags[4]).toBe('datatype=d');
  expect(frags[5]).toBe('interval=e');
});

test(`the url builder with no params yields the base url`, () => {
  expect.assertions(3);
  const frags = alpha.util
    .url()
    .toString()
    .split('?')[1]
    .toString()
    .split('&');

  expect(frags.length).toBe(2);
  expect(/^apikey=.*$/.test(frags[0])).toBe(true);
  expect(frags[1]).toBe('');
});

test(`the url builder with undefined params yields the base url`, () => {
  expect.assertions(3);
  const frags = alpha.util
    .url({ 'this_does-not_exist': undefined })
    .toString()
    .split('?')[1]
    .toString()
    .split('&');

  expect(frags.length).toBe(2);
  expect(/^apikey=.*$/.test(frags[0])).toBe(true);
  expect(frags[1]).toBe('');
});

test(`intraday data polishing works`, () => {
  expect.assertions(41);
  const data = require('./examples/data/intraday.json');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Symbol']).toBeDefined();
  expect(data['Meta Data']['3. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['4. Interval']).toBeDefined();
  expect(data['Meta Data']['5. Output Size']).toBeDefined();
  expect(data['Meta Data']['6. Time Zone']).toBeDefined();
  expect(data['Time Series (1min)']).toBeDefined();
  first = Object.keys(data['Time Series (1min)'])[0];
  expect(first).toBeDefined();
  expect(data['Time Series (1min)'][first]['1. open']).toBeDefined();
  expect(data['Time Series (1min)'][first]['2. high']).toBeDefined();
  expect(data['Time Series (1min)'][first]['3. low']).toBeDefined();
  expect(data['Time Series (1min)'][first]['4. close']).toBeDefined();
  expect(data['Time Series (1min)'][first]['5. volume']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (1min)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Symbol']).toBeUndefined();
  expect(polished['meta']['3. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['4. Interval']).toBeUndefined();
  expect(polished['meta']['5. Output Size']).toBeUndefined();
  expect(polished['meta']['6. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['symbol']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['interval']).toBeDefined();
  expect(polished['meta']['size']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1. open']).toBeUndefined();
  expect(polished['data'][first]['2. high']).toBeUndefined();
  expect(polished['data'][first]['3. low']).toBeUndefined();
  expect(polished['data'][first]['4. close']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['open']).toBeDefined();
  expect(polished['data'][first]['high']).toBeDefined();
  expect(polished['data'][first]['low']).toBeDefined();
  expect(polished['data'][first]['close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
});

test(`daily data polishing works`, () => {
  expect.assertions(38);
  const data = require('./examples/data/daily.json');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Symbol']).toBeDefined();
  expect(data['Meta Data']['3. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['4. Output Size']).toBeDefined();
  expect(data['Meta Data']['5. Time Zone']).toBeDefined();
  expect(data['Time Series (Daily)']).toBeDefined();
  first = Object.keys(data['Time Series (Daily)'])[0];
  expect(first).toBeDefined();
  expect(data['Time Series (Daily)'][first]['1. open']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['2. high']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['3. low']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['4. close']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['5. volume']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (Daily)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Symbol']).toBeUndefined();
  expect(polished['meta']['3. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['4. Output Size']).toBeUndefined();
  expect(polished['meta']['5. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['symbol']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['size']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1. open']).toBeUndefined();
  expect(polished['data'][first]['2. high']).toBeUndefined();
  expect(polished['data'][first]['3. low']).toBeUndefined();
  expect(polished['data'][first]['4. close']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['open']).toBeDefined();
  expect(polished['data'][first]['high']).toBeDefined();
  expect(polished['data'][first]['low']).toBeDefined();
  expect(polished['data'][first]['close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
});

test(`adjusted data polishing works`, () => {
  expect.assertions(47);
  const data = require('./examples/data/adjusted.json');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Symbol']).toBeDefined();
  expect(data['Meta Data']['3. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['4. Output Size']).toBeDefined();
  expect(data['Meta Data']['5. Time Zone']).toBeDefined();
  expect(data['Time Series (Daily)']).toBeDefined();
  first = Object.keys(data['Time Series (Daily)'])[0];
  expect(first).toBeDefined();
  expect(data['Time Series (Daily)'][first]['1. open']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['2. high']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['3. low']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['4. close']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['5. adjusted close']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['6. volume']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['7. dividend amount']).toBeDefined();
  expect(data['Time Series (Daily)'][first]['8. split coefficient']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (Daily)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Symbol']).toBeUndefined();
  expect(polished['meta']['3. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['4. Output Size']).toBeUndefined();
  expect(polished['meta']['5. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['symbol']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['size']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1. open']).toBeUndefined();
  expect(polished['data'][first]['2. high']).toBeUndefined();
  expect(polished['data'][first]['3. low']).toBeUndefined();
  expect(polished['data'][first]['4. close']).toBeUndefined();
  expect(polished['data'][first]['5. adjusted close']).toBeUndefined();
  expect(polished['data'][first]['6. volume']).toBeUndefined();
  expect(polished['data'][first]['7. dividend amount']).toBeUndefined();
  expect(polished['data'][first]['8. split coefficient']).toBeUndefined();
  expect(polished['data'][first]['open']).toBeDefined();
  expect(polished['data'][first]['high']).toBeDefined();
  expect(polished['data'][first]['low']).toBeDefined();
  expect(polished['data'][first]['close']).toBeDefined();
  expect(polished['data'][first]['adjusted']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
  expect(polished['data'][first]['dividend']).toBeDefined();
  expect(polished['data'][first]['split']).toBeDefined();
});

test(`weekly data polishing works`, () => {
  expect.assertions(35);
  const data = require('./examples/data/weekly.json');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Symbol']).toBeDefined();
  expect(data['Meta Data']['3. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['4. Time Zone']).toBeDefined();
  expect(data['Weekly Time Series']).toBeDefined();
  first = Object.keys(data['Weekly Time Series'])[0];
  expect(first).toBeDefined();
  expect(data['Weekly Time Series'][first]['1. open']).toBeDefined();
  expect(data['Weekly Time Series'][first]['2. high']).toBeDefined();
  expect(data['Weekly Time Series'][first]['3. low']).toBeDefined();
  expect(data['Weekly Time Series'][first]['4. close']).toBeDefined();
  expect(data['Weekly Time Series'][first]['5. volume']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Weekly Time Series']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Symbol']).toBeUndefined();
  expect(polished['meta']['3. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['4. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['symbol']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1. open']).toBeUndefined();
  expect(polished['data'][first]['2. high']).toBeUndefined();
  expect(polished['data'][first]['3. low']).toBeUndefined();
  expect(polished['data'][first]['4. close']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['open']).toBeDefined();
  expect(polished['data'][first]['high']).toBeDefined();
  expect(polished['data'][first]['low']).toBeDefined();
  expect(polished['data'][first]['close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
});

test(`monthly data polishing works`, () => {
  expect.assertions(35);
  const data = require('./examples/data/monthly.json');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Symbol']).toBeDefined();
  expect(data['Meta Data']['3. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['4. Time Zone']).toBeDefined();
  expect(data['Monthly Time Series']).toBeDefined();
  first = Object.keys(data['Monthly Time Series'])[0];
  expect(first).toBeDefined();
  expect(data['Monthly Time Series'][first]['1. open']).toBeDefined();
  expect(data['Monthly Time Series'][first]['2. high']).toBeDefined();
  expect(data['Monthly Time Series'][first]['3. low']).toBeDefined();
  expect(data['Monthly Time Series'][first]['4. close']).toBeDefined();
  expect(data['Monthly Time Series'][first]['5. volume']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Monthly Time Series']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Symbol']).toBeUndefined();
  expect(polished['meta']['3. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['4. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['symbol']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1. open']).toBeUndefined();
  expect(polished['data'][first]['2. high']).toBeUndefined();
  expect(polished['data'][first]['3. low']).toBeUndefined();
  expect(polished['data'][first]['4. close']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['open']).toBeDefined();
  expect(polished['data'][first]['high']).toBeDefined();
  expect(polished['data'][first]['low']).toBeDefined();
  expect(polished['data'][first]['close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
});

test(`forex rate data polishing works`, () => {
  expect.assertions(24);
  const data = require('./examples/forex/rate.json');
  const polished = alpha.util.polish(data);

  expect(data['Realtime Currency Exchange Rate']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['1. From_Currency Code']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['2. From_Currency Name']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['3. To_Currency Code']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['4. To_Currency Name']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['5. Exchange Rate']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['6. Last Refreshed']).toBeDefined();
  expect(data['Realtime Currency Exchange Rate']['7. Time Zone']).toBeDefined();

  expect(polished['Realtime Currency Exchange Rate']).toBeUndefined();
  expect(polished['rate']['1. From_Currency Code']).toBeUndefined();
  expect(polished['rate']['2. From_Currency Name']).toBeUndefined();
  expect(polished['rate']['3. To_Currency Code']).toBeUndefined();
  expect(polished['rate']['4. To_Currency Name']).toBeUndefined();
  expect(polished['rate']['5. Exchange Rate']).toBeUndefined();
  expect(polished['rate']['6. Last Refreshed']).toBeUndefined();
  expect(polished['rate']['7. Time Zone']).toBeUndefined();

  expect(polished['rate']).toBeDefined();
  expect(polished['rate']['from_currency']).toBeDefined();
  expect(polished['rate']['from_currency_name']).toBeDefined();
  expect(polished['rate']['to_currency']).toBeDefined();
  expect(polished['rate']['to_currency_name']).toBeDefined();
  expect(polished['rate']['value']).toBeDefined();
  expect(polished['rate']['updated']).toBeDefined();
  expect(polished['rate']['zone']).toBeDefined();
});

test(`intraday crypto polishing works`, () => {
  expect.assertions(44);
  const data = require('./examples/crypto/intraday');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Digital Currency Code']).toBeDefined();
  expect(data['Meta Data']['3. Digital Currency Name']).toBeDefined();
  expect(data['Meta Data']['4. Market Code']).toBeDefined();
  expect(data['Meta Data']['5. Market Name']).toBeDefined();
  expect(data['Meta Data']['6. Interval']).toBeDefined();
  expect(data['Meta Data']['7. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['8. Time Zone']).toBeDefined();
  expect(data['Time Series (Digital Currency Intraday)']).toBeDefined();
  first = Object.keys(data['Time Series (Digital Currency Intraday)'])[0];
  expect(first).toBeDefined();
  expect(data['Time Series (Digital Currency Intraday)'][first]['1a. price (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Intraday)'][first]['1b. price (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Intraday)'][first]['2. volume']).toBeDefined();
  expect(data['Time Series (Digital Currency Intraday)'][first]['3. market cap (USD)']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (Digital Currency Intraday)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Digital Currency Code']).toBeUndefined();
  expect(polished['meta']['3. Digital Currency Name']).toBeUndefined();
  expect(polished['meta']['4. Market Code']).toBeUndefined();
  expect(polished['meta']['5. Market Name']).toBeUndefined();
  expect(polished['meta']['6. Interval']).toBeUndefined();
  expect(polished['meta']['7. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['8. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['coin']).toBeDefined();
  expect(polished['meta']['coin_name']).toBeDefined();
  expect(polished['meta']['market']).toBeDefined();
  expect(polished['meta']['market_name']).toBeDefined();
  expect(polished['meta']['interval']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1a. price (CNY)']).toBeUndefined();
  expect(polished['data'][first]['1b. price (USD)']).toBeUndefined();
  expect(polished['data'][first]['2. volume']).toBeUndefined();
  expect(polished['data'][first]['3. market cap (USD)']).toBeUndefined();
  expect(polished['data'][first]['market']).toBeDefined();
  expect(polished['data'][first]['usd']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
  expect(polished['data'][first]['cap']).toBeDefined();
});

test(`daily crypto polishing works`, () => {
  expect.assertions(59);
  const data = require('./examples/crypto/daily');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Digital Currency Code']).toBeDefined();
  expect(data['Meta Data']['3. Digital Currency Name']).toBeDefined();
  expect(data['Meta Data']['4. Market Code']).toBeDefined();
  expect(data['Meta Data']['5. Market Name']).toBeDefined();
  expect(data['Meta Data']['6. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['7. Time Zone']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)']).toBeDefined();
  first = Object.keys(data['Time Series (Digital Currency Daily)'])[0];

  expect(first).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['1a. open (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['1b. open (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['2a. high (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['2b. high (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['3a. low (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['3b. low (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['4a. close (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['4b. close (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['5. volume']).toBeDefined();
  expect(data['Time Series (Digital Currency Daily)'][first]['6. market cap (USD)']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (Digital Currency Daily)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Digital Currency Code']).toBeUndefined();
  expect(polished['meta']['3. Digital Currency Name']).toBeUndefined();
  expect(polished['meta']['4. Market Code']).toBeUndefined();
  expect(polished['meta']['5. Market Name']).toBeUndefined();
  expect(polished['meta']['6. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['7. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['coin']).toBeDefined();
  expect(polished['meta']['coin_name']).toBeDefined();
  expect(polished['meta']['market']).toBeDefined();
  expect(polished['meta']['market_name']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1a. open (CNY)']).toBeUndefined();
  expect(polished['data'][first]['1b. open (USD)']).toBeUndefined();
  expect(polished['data'][first]['2a. high (CNY)']).toBeUndefined();
  expect(polished['data'][first]['2b. high (USD)']).toBeUndefined();
  expect(polished['data'][first]['3a. low (CNY)']).toBeUndefined();
  expect(polished['data'][first]['3b. low (USD)']).toBeUndefined();
  expect(polished['data'][first]['4a. close (CNY)']).toBeUndefined();
  expect(polished['data'][first]['4b. close (USD)']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['6. market cap (USD)']).toBeUndefined();
  expect(polished['data'][first]['market_open']).toBeDefined();
  expect(polished['data'][first]['market_high']).toBeDefined();
  expect(polished['data'][first]['market_low']).toBeDefined();
  expect(polished['data'][first]['market_close']).toBeDefined();
  expect(polished['data'][first]['usd_open']).toBeDefined();
  expect(polished['data'][first]['usd_high']).toBeDefined();
  expect(polished['data'][first]['usd_low']).toBeDefined();
  expect(polished['data'][first]['usd_close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
  expect(polished['data'][first]['cap']).toBeDefined();
});

test(`weekly crypto polishing works`, () => {
  expect.assertions(59);
  const data = require('./examples/crypto/weekly');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Digital Currency Code']).toBeDefined();
  expect(data['Meta Data']['3. Digital Currency Name']).toBeDefined();
  expect(data['Meta Data']['4. Market Code']).toBeDefined();
  expect(data['Meta Data']['5. Market Name']).toBeDefined();
  expect(data['Meta Data']['6. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['7. Time Zone']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)']).toBeDefined();
  first = Object.keys(data['Time Series (Digital Currency Weekly)'])[0];

  expect(first).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['1a. open (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['1b. open (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['2a. high (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['2b. high (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['3a. low (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['3b. low (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['4a. close (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['4b. close (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['5. volume']).toBeDefined();
  expect(data['Time Series (Digital Currency Weekly)'][first]['6. market cap (USD)']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (Digital Currency Weekly)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Digital Currency Code']).toBeUndefined();
  expect(polished['meta']['3. Digital Currency Name']).toBeUndefined();
  expect(polished['meta']['4. Market Code']).toBeUndefined();
  expect(polished['meta']['5. Market Name']).toBeUndefined();
  expect(polished['meta']['6. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['7. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['coin']).toBeDefined();
  expect(polished['meta']['coin_name']).toBeDefined();
  expect(polished['meta']['market']).toBeDefined();
  expect(polished['meta']['market_name']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1a. open (CNY)']).toBeUndefined();
  expect(polished['data'][first]['1b. open (USD)']).toBeUndefined();
  expect(polished['data'][first]['2a. high (CNY)']).toBeUndefined();
  expect(polished['data'][first]['2b. high (USD)']).toBeUndefined();
  expect(polished['data'][first]['3a. low (CNY)']).toBeUndefined();
  expect(polished['data'][first]['3b. low (USD)']).toBeUndefined();
  expect(polished['data'][first]['4a. close (CNY)']).toBeUndefined();
  expect(polished['data'][first]['4b. close (USD)']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['6. market cap (USD)']).toBeUndefined();
  expect(polished['data'][first]['market_open']).toBeDefined();
  expect(polished['data'][first]['market_high']).toBeDefined();
  expect(polished['data'][first]['market_low']).toBeDefined();
  expect(polished['data'][first]['market_close']).toBeDefined();
  expect(polished['data'][first]['usd_open']).toBeDefined();
  expect(polished['data'][first]['usd_high']).toBeDefined();
  expect(polished['data'][first]['usd_low']).toBeDefined();
  expect(polished['data'][first]['usd_close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
  expect(polished['data'][first]['cap']).toBeDefined();
});

test(`monthly crypto polishing works`, () => {
  expect.assertions(59);
  const data = require('./examples/crypto/monthly');
  const polished = alpha.util.polish(data);
  let first;

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['1. Information']).toBeDefined();
  expect(data['Meta Data']['2. Digital Currency Code']).toBeDefined();
  expect(data['Meta Data']['3. Digital Currency Name']).toBeDefined();
  expect(data['Meta Data']['4. Market Code']).toBeDefined();
  expect(data['Meta Data']['5. Market Name']).toBeDefined();
  expect(data['Meta Data']['6. Last Refreshed']).toBeDefined();
  expect(data['Meta Data']['7. Time Zone']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)']).toBeDefined();
  first = Object.keys(data['Time Series (Digital Currency Monthly)'])[0];

  expect(first).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['1a. open (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['1b. open (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['2a. high (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['2b. high (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['3a. low (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['3b. low (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['4a. close (CNY)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['4b. close (USD)']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['5. volume']).toBeDefined();
  expect(data['Time Series (Digital Currency Monthly)'][first]['6. market cap (USD)']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Time Series (Digital Currency Monthly)']).toBeUndefined();
  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['1. Information']).toBeUndefined();
  expect(polished['meta']['2. Digital Currency Code']).toBeUndefined();
  expect(polished['meta']['3. Digital Currency Name']).toBeUndefined();
  expect(polished['meta']['4. Market Code']).toBeUndefined();
  expect(polished['meta']['5. Market Name']).toBeUndefined();
  expect(polished['meta']['6. Last Refreshed']).toBeUndefined();
  expect(polished['meta']['7. Time Zone']).toBeUndefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['coin']).toBeDefined();
  expect(polished['meta']['coin_name']).toBeDefined();
  expect(polished['meta']['market']).toBeDefined();
  expect(polished['meta']['market_name']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['meta']['zone']).toBeDefined();
  expect(polished['data']).toBeDefined();
  first = Object.keys(polished['data'])[0];
  expect(first).toBeDefined();
  expect(polished['data'][first]['1a. open (CNY)']).toBeUndefined();
  expect(polished['data'][first]['1b. open (USD)']).toBeUndefined();
  expect(polished['data'][first]['2a. high (CNY)']).toBeUndefined();
  expect(polished['data'][first]['2b. high (USD)']).toBeUndefined();
  expect(polished['data'][first]['3a. low (CNY)']).toBeUndefined();
  expect(polished['data'][first]['3b. low (USD)']).toBeUndefined();
  expect(polished['data'][first]['4a. close (CNY)']).toBeUndefined();
  expect(polished['data'][first]['4b. close (USD)']).toBeUndefined();
  expect(polished['data'][first]['5. volume']).toBeUndefined();
  expect(polished['data'][first]['6. market cap (USD)']).toBeUndefined();
  expect(polished['data'][first]['market_open']).toBeDefined();
  expect(polished['data'][first]['market_high']).toBeDefined();
  expect(polished['data'][first]['market_low']).toBeDefined();
  expect(polished['data'][first]['market_close']).toBeDefined();
  expect(polished['data'][first]['usd_open']).toBeDefined();
  expect(polished['data'][first]['usd_high']).toBeDefined();
  expect(polished['data'][first]['usd_low']).toBeDefined();
  expect(polished['data'][first]['usd_close']).toBeDefined();
  expect(polished['data'][first]['volume']).toBeDefined();
  expect(polished['data'][first]['cap']).toBeDefined();
});

test(`sector performance data polishing works`, () => {
  expect.assertions(48);
  const data = require('./examples/sector/performance.json');
  const polished = alpha.util.polish(data);

  expect(data['Meta Data']).toBeDefined();
  expect(data['Meta Data']['Information']).toBeDefined();
  expect(data['Meta Data']['Last Refreshed']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']).toBeDefined();
  expect(data['Rank B: 1 Day Performance']).toBeDefined();
  expect(data['Rank C: 5 Day Performance']).toBeDefined();
  expect(data['Rank D: 1 Month Performance']).toBeDefined();
  expect(data['Rank E: 3 Month Performance']).toBeDefined();
  expect(data['Rank F: Year-to-Date (YTD) Performance']).toBeDefined();
  expect(data['Rank G: 1 Year Performance']).toBeDefined();
  expect(data['Rank H: 3 Year Performance']).toBeDefined();
  expect(data['Rank I: 5 Year Performance']).toBeDefined();
  expect(data['Rank J: 10 Year Performance']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Information Technology']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Consumer Discretionary']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Health Care']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Industrials']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Consumer Staples']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Telecommunication Services']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Materials']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Financials']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Real Estate']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Utilities']).toBeDefined();
  expect(data['Rank A: Real-Time Performance']['Energy']).toBeDefined();

  expect(polished['Meta Data']).toBeUndefined();
  expect(polished['Rank A: Real-Time Performance']).toBeUndefined();
  expect(polished['Rank B: 1 Day Performance']).toBeUndefined();
  expect(polished['Rank C: 5 Day Performance']).toBeUndefined();
  expect(polished['Rank D: 1 Month Performance']).toBeUndefined();
  expect(polished['Rank E: 3 Month Performance']).toBeUndefined();
  expect(polished['Rank F: Year-to-Date (YTD) Performance']).toBeUndefined();
  expect(polished['Rank G: 1 Year Performance']).toBeUndefined();
  expect(polished['Rank H: 3 Year Performance']).toBeUndefined();
  expect(polished['Rank I: 5 Year Performance']).toBeUndefined();
  expect(polished['Rank J: 10 Year Performance']).toBeUndefined();

  expect(polished['meta']).toBeDefined();
  expect(polished['meta']['information']).toBeDefined();
  expect(polished['meta']['updated']).toBeDefined();
  expect(polished['real']).toBeDefined();
  expect(polished['1day']).toBeDefined();
  expect(polished['5day']).toBeDefined();
  expect(polished['1month']).toBeDefined();
  expect(polished['3month']).toBeDefined();
  expect(polished['ytd']).toBeDefined();
  expect(polished['1year']).toBeDefined();
  expect(polished['3year']).toBeDefined();
  expect(polished['5year']).toBeDefined();
  expect(polished['10year']).toBeDefined();
});

test(`non 200 request responses are thrown to a catch`, () => {
  expect.assertions(1);

  return alpha.util.fn('123')().catch(error => {
    expect(error).toEqual('An AlphaVantage error occurred. 123: {}');
  });
});

test(`200 request responses without meta data are thrown to a catch`, () => {
  expect.assertions(1);

  return alpha.util.fn('200')().catch(error => {
    expect(error).toEqual('An AlphaVantage error occurred. {}');
  });
});
